#!/usr/bin/env python3

import os
import re
import subprocess
import sys
from datetime import datetime

tmpcert = "/tmp/tmpcert.der"

incert = False
intrust = False

if len(sys.argv) != 2:
    print()
    print("Pass path to the certdata.txt you want to add as first input argument to")
    print("this script certdata.txt can be retrieved from the newest NSS release at")
    print("https://ftp.mozilla.org/pub/security/nss/releases/.")
    sys.exit(1)

with open(sys.argv[1]) as f:
    certdata = f.readlines()

tz = datetime.now().astimezone().tzinfo

print(
    "# Do not edit this file. This file is automatically generated by gen-mozilla-ca-list.py"
)
print(f"# Generated at: {datetime.now(tz=tz).strftime('%Y-%m-%d %H:%M:%S %z')}")
print("# Generated from: #REPLACE WITH NSS VERSION#")
print("#")
print("# The original source file comes with this licensing statement:")
print("#")
print("#   This Source Code Form is subject to the terms of the Mozilla Public")
print("#   License, v. 2.0. If a copy of the MPL was not distributed with this")
print("#   file, You can obtain one at https://mozilla.org/MPL/2.0/.")
print("")

all_certs = []
all_subjects = []

print("@load base/protocols/ssl")
print("module SSL;")
print("")
print("## @docs-omit-value")
print("redef root_certs += {")

cert = ""
for line in certdata:
    line = line.strip()

    if intrust:
        if line.startswith("CKA_TRUST_SERVER_AUTH"):
            if line.find("CKT_NSS_TRUSTED_DELEGATOR") != -1:
                with open(tmpcert, "wb") as f:
                    bytearray = [int(b, 16) for b in cert.split("\\x")[1:]]
                    f.write(bytes(bytearray))

                result = subprocess.run(
                    [
                        "openssl",
                        "x509",
                        "-in",
                        tmpcert,
                        "-inform",
                        "DER",
                        "-noout",
                        "-subject",
                        "-nameopt",
                        "RFC2253",
                    ],
                    capture_output=True,
                    text=True,
                )
                cert_subj = result.stdout.replace("subject=", "").strip()
                if os.path.exists(tmpcert):
                    try:
                        os.remove(tmpcert)
                    except:
                        print(f"Failed to remove {tmpcert}")

                if cert_subj not in all_subjects:
                    print(f'\t["{cert_subj}"] = "{cert}",')
                    all_subjects.append(cert_subj)
            intrust = False
    else:
        if line.startswith("CKA_LABEL"):
            m = re.match(r".*\"(.*)\".*", line)
            if m:
                cert_name = m.group(1)
                if cert_name not in all_certs:
                    cert_name += f" #{len(all_certs) + 1}"
                all_certs.append(cert_name)
        elif line.startswith("CKA_VALUE MULTILINE_OCTAL"):
            incert = True
            cert = ""
        elif line.startswith("CKA_CLASS CK_OBJECT_CLASS CKO_NSS_TRUST"):
            intrust = True
        elif line.startswith("END"):
            incert = False
        elif incert:
            octals = line.split("\\")[1:]
            hexes = [hex(int(x, 8))[2:].zfill(2).upper() for x in octals]
            cert += "\\x".join(["", *hexes])

print("};")
